//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2023 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#import "GameViewController.h"
#import "RenderBridge.h"

@implementation GameViewController
{
    MTKView *_view;
    RenderBridge *_renderer;
    RenderMode _currentRenderMode;
    CGFloat _configPanelH;
}

#pragma region - Command-line configuration
- (void)evaluateCommandLine
{
    NSArray<NSString *>* args = [[NSProcessInfo processInfo] arguments];
    BOOL exitAfterOneFrame = [args containsObject:@"--auto-close"];
    BOOL noAnimation       = [args containsObject:@"--no-animation"];
    BOOL defaultToGSMode   = [args containsObject:@"--default-rendermode-gs"];
    BOOL defaultToFlatMode = [args containsObject:@"--default-rendermode-flat"];
    BOOL nonIndexedTSMode  = [args containsObject:@"--default-rendermode-non-indexed-ts"];
    BOOL nonIndexedGSMode  = [args containsObject:@"--default-rendermode-non-indexed-gs"];
    
    if (exitAfterOneFrame)
    {
        NSLog(@"Automatically terminating in 5 seconds...");
        [[NSApplication sharedApplication] performSelector:@selector(terminate:) withObject:self afterDelay:5];
    }
    
    if (noAnimation)
    {
        NSLog(@"Set initial animation speed to 0");
        kDefaultAnimationFactor = 0;
    }
    
    if (defaultToFlatMode)
    {
        NSLog(@"Set default grass render mode to flat triangles");
        kDefaultRenderMode = RM_NoGS_NoTS;
    }
    
    if (defaultToGSMode)
    {
        NSLog(@"Set default grass render mode to Geometry Shader");
        kDefaultRenderMode = RM_GS_NoTS;
    }
    
    if (nonIndexedTSMode)
    {
        NSLog(@"Set default grass render mode to non-indexed tessellation");
        kDefaultRenderMode = RM_GS_TS_NonIndexed;
    }
    
    if (nonIndexedGSMode)
    {
        NSLog(@"Set default grass render mode to non-indexed Geometry Shader");
        kDefaultRenderMode = RM_GS_NoTS_NonIndexed;
    }
    
}

#pragma region - UI configuration

- (void)viewDidLoad
{
    [super viewDidLoad];
    [self evaluateCommandLine];
    
    [self _configureBackdrop:_configBackdrop];
    
    _animationFactorSlider.floatValue = kDefaultAnimationFactor;
    _tessellationInnerSlider.floatValue = kDefaultTessellationInnerValue;
    _currentRenderMode = kDefaultRenderMode;
    
    // Decode UI state from command-line args:
    if (_currentRenderMode == RM_GS_TS || _currentRenderMode == RM_GS_NoTS || _currentRenderMode == RM_NoGS_NoTS)
    {
        _renderModeControl.selectedSegment = _currentRenderMode;
    }
    else
    {
        _indexedDrawToggle.state = NSControlStateValueOff;
        if (_currentRenderMode == RM_GS_NoTS_NonIndexed)
        {
            _renderModeControl.selectedSegment = RM_GS_NoTS;
        }
        else if (_currentRenderMode == RM_GS_TS_NonIndexed)
        {
            _renderModeControl.selectedSegment = RM_GS_TS;
        }
    }
    
    _view = (MTKView *)self.view;
    
    _view.device = MTLCreateSystemDefaultDevice();
    _view.colorPixelFormat = MTLPixelFormatBGRA8Unorm_sRGB;
    _view.depthStencilPixelFormat = MTLPixelFormatDepth16Unorm;
    _view.clearColor = MTLClearColorMake( 0.53, 0.81, 0.92, 1.0 );
    _view.clearDepth = 1.0f;
    _view.depthStencilStorageMode = MTLStorageModeMemoryless;
    
    if(!_view.device)
    {
        NSLog(@"Metal is not supported on this device");
        self.view = [[NSView alloc] initWithFrame:self.view.frame];
        return;
    }
    
    // Compile shaders in a secondary high-priority thread to prevent UI stalls:
    [_progressIndicator startAnimation:self];
    
    CGSize boundsSize = _view.bounds.size;
    __block GameViewController* weakSelf = self;
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_HIGH, 0), ^(){
        
        GameViewController* strongSelf = weakSelf;
        
        strongSelf->_renderer = [[RenderBridge alloc] initWithMetalKitView:self->_view];
        [strongSelf->_renderer mtkView:self->_view drawableSizeWillChange:boundsSize];
        strongSelf->_view.delegate = self->_renderer;
        
        dispatch_async(dispatch_get_main_queue(), ^(){
            [strongSelf->_progressIndicator stopAnimation:self];
            [strongSelf->_progressIndicator setHidden:YES];
        });
    });
}

- (void)viewWillAppear
{
    _configPanelH = _configBackdrop.frame.size.height;
    [self _updateUIControlsForRenderMode:_currentRenderMode];
}

- (void)updateWindowTitleAndTargetFPS
{
    NSScreen* screen = _view.window.screen;
    NSString* title = [NSString stringWithFormat:@"Metal Shader Converter Dylib Example (%@ @ %ldHz)", screen.localizedName, (long)screen.maximumFramesPerSecond];
    _view.window.title = title;
    _view.preferredFramesPerSecond = screen.maximumFramesPerSecond;
    _view.window.delegate = self;
}

- (void)viewDidAppear
{
    [self updateWindowTitleAndTargetFPS];
}

- (void)windowDidChangeScreen:(NSNotification *)notification
{
    [self updateWindowTitleAndTargetFPS];
}

- (void)_configureBackdrop:(NSView *)view
{
    view.wantsLayer = YES;
    view.layer.borderWidth = 1.0f;
    view.layer.cornerRadius = 8.0f;
}

#pragma region - UI callbacks

- (void)_updateUIControlsForRenderMode:(RenderMode)next
{
    RenderMode index = (RenderMode)_renderModeControl.indexOfSelectedItem;
    
    // Hide tessellation controls for nontessellation modes.
    
    float tessControlAlpha = (next != RM_GS_TS) ? 0.0f : 1.0f;
    _tessellationInnerSliderLabel.animator.alphaValue = tessControlAlpha;
    _tessellationInnerSlider.animator.alphaValue = tessControlAlpha;
    
    // Adjust the height of the control backdrop:
    
    static CGFloat panelHeightForMode[3] = { -50, -50, 0 };
    
    NSRect backdropRect = _configBackdrop.frame;
    backdropRect.size.height = _configPanelH + panelHeightForMode[next];
    _configBackdrop.animator.frame = backdropRect;
    
    _currentRenderMode = index;
}

- (IBAction)onRenderModeSegmentedControlAction:(id)sender
{
    if ( sender == _renderModeControl )
    {
        RenderMode index = (RenderMode)_renderModeControl.indexOfSelectedItem;
        if ( index != _currentRenderMode )
        {
            [self _updateUIControlsForRenderMode:(RenderMode)_renderModeControl.indexOfSelectedItem];
        }
        _currentRenderMode = (RenderMode)_renderModeControl.indexOfSelectedItem;
        _renderer.renderMode = index;
        
        // Take into consideration state of indexed draw toggle
        if (_currentRenderMode == RM_GS_TS)
        {
            if (_indexedDrawToggle.state == NSControlStateValueOff)
            {
                _currentRenderMode = RM_GS_TS_NonIndexed;
                _renderer.renderMode = _currentRenderMode;
            }
        }
        else if (_currentRenderMode == RM_GS_NoTS)
        {
            if (_indexedDrawToggle.state == NSControlStateValueOff)
            {
                _currentRenderMode = RM_GS_NoTS_NonIndexed;
                _renderer.renderMode = _currentRenderMode;
            }
        }
    }
}

- (IBAction)onIndexedDrawToggleAction:(id)sender
{
    NSButton* checkbox = (NSButton *)sender;
    BOOL enableIndexed = checkbox.state == NSControlStateValueOn;
    
    if (_currentRenderMode == RM_GS_TS || _currentRenderMode == RM_GS_TS_NonIndexed)
    {
        _currentRenderMode = (enableIndexed) ? RM_GS_TS : RM_GS_TS_NonIndexed;
    }
    else if (_currentRenderMode == RM_GS_NoTS || _currentRenderMode == RM_GS_NoTS_NonIndexed)
    {
        _currentRenderMode = (enableIndexed) ? RM_GS_NoTS : RM_GS_NoTS_NonIndexed;
    }
    
    _renderer.renderMode = _currentRenderMode;
}

- (IBAction)onCheckboxToggledAction:(id)sender
{
    NSButton* checkbox = (NSButton *)sender;
    _renderer.triangleFillMode = (checkbox.state == NSControlStateValueOn) ? MTLTriangleFillModeLines : MTLTriangleFillModeFill;
}

- (IBAction)onAnimationSpeedSliderValueChanged:(id)sender
{
    NSSlider* slider = (NSSlider *)sender;
    _renderer.animationFactor = slider.floatValue;
}

- (IBAction)onInnerTessellationSliderValueChanged:(id)sender
{
    NSSlider* slider = (NSSlider *)sender;
    _renderer.tessellationInnerValue = slider.floatValue;
}

@end
